/***************************************************************
 *                 Mathematical Object Library                 *
 *           Conversion tools Library - Include File           *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.3.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2003,2004,2005,2006 COLLARD Christophe
 * copyright  2003,2004,2005,2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2003,2004,2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class conversion_tools
    \brief conversion tools library for basic mathematical objects \n

    \htmlonly 
    <FONT color="#838383">

    conversion_tools belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2003, 2004, 2005, 2006 Christophe COLLARD \n
	    copyright \htmlonly &#169; \endhtmlonly 2003, 2004, 2005, 2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	    copyright \htmlonly &#169; \endhtmlonly 2003, 2004, 2005, 2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.3.0
    \date 2003-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type conversion tools
#endif

#ifndef _conversion_tools_h
#define _conversion_tools_h

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__PRECISION_H)
#include "precision.h"
#endif

#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "../MOL++/matrix.h"
#endif

#if !defined(__TENSORS4_H)
#include "../MOL++/tensors4.h"
#endif

using namespace std;


//====================
class conversion_tools
//====================
{
  public :
    template <class T> friend matrix<T>  vector2matrix    (const vector<T>&, int, int);
    template <class T> friend tensor2<T> vector2symtensor (const vector<T>&);
    template <class T> friend vector<T>  matrix2vector    (const matrix<T>&);
    template <class T> friend tensor4<T> matrix2tensor    (const matrix<T>&, int, int, int, int);
    template <class T> friend tensor4<T> matrix2symtensor (const matrix<T>&);
    template <class T> friend matrix<T>  tensor2matrix    (const tensor4<T>&);
    template <class T> friend vector<T>  symtensor2vector (const tensor2<T>&);
    template <class T> friend matrix<T>  symtensor2matrix (const tensor4<T>&);
};


//------------------------------------------------------------------------------------
template <class T> matrix<T> vector2matrix (const vector<T>& v, int rows, int columns)
//------------------------------------------------------------------------------------
{
  assert (v.dim() == rows*columns);

  matrix<T> mat(rows,columns,false);

  for (int i=1,p=1; i<=rows; i++)
    for (int j=1; j<=columns; j++)
      mat(i,j) = v[p++];

  return mat;
}


//-----------------------------------------------------------------
template <class T> tensor2<T> vector2symtensor (const vector<T>& v)
//-----------------------------------------------------------------
{
  assert (v.dim());

  long double factor = .5 * sqrt( (long double) 2);
  int n = .5 * (sqrt(1.+8*v.dim()) - 1);
  tensor2<T> tsr(n,n,false);

  for (int i=1,p=1; i<=n; i++)
    { tsr(i,i) = v[p++];
      for (int j=i+1; j<=n; j++)
	tsr(i,j) = tsr(j,i) = factor * v[p++];
    }

  return tsr;
}


//---------------------------------------------------------------
template <class T> vector<T> matrix2vector (const matrix<T>& mat)
//---------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());

  vector<T> v(mat.Rows()*mat.Columns(),false);

  for (int i=1,p=1; i<=mat.Rows(); i++)
    for (int j=1; j<=mat.Columns(); j++)
      v[p++] = mat(i,j);

  return v;
}


//------------------------------------------------------------------------------------------------
template <class T> tensor4<T> matrix2tensor (const matrix<T>& mat, int d1, int d2, int d3, int d4)
//------------------------------------------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());
  assert ( (d1*d2 == mat.Rows()) && (d3*d4 == mat.Columns()) );

  tensor4<T> tsr(d1,d2,d3,d4,false);

  for (int i=1,p=1; i<=tsr.dim1(); i++)
    for (int j=1; j<=tsr.dim2(); j++,p++)
      for (int k=1,q=1; k<=tsr.dim3(); k++)
	for (int l=1; l<=tsr.dim4(); l++)
	  tsr(i,j,k,l) = mat (p,q++);

  return tsr;
}


//-------------------------------------------------------------------
template <class T> tensor4<T> matrix2symtensor (const matrix<T>& mat)
//-------------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());

  long double factor = .5 * sqrt( (long double) 2);
  int nb1 = .5 * (sqrt(1.+8*mat.Rows()) - 1);
  int nb2 = .5 * (sqrt(1.+8*mat.Columns()) - 1);
  tensor4<T> tsr(nb1,nb1,nb2,nb2,false);

  for (int i=1,p=1; i<=nb1; i++)
    for (int j=i; j<=nb1; j++,p++)
      for (int k=1,q=1; k<=nb2; k++)
	for (int l=k; l<=nb2; l++)
	  if (i==j && k==l) tsr(i,j,k,l) = mat(p,q++);
	  else if (i!=j && k!=l) tsr(i,j,k,l) = tsr(j,i,k,l) = tsr(i,j,l,k) = tsr(j,i,l,k) = 0.5 * mat(p,q++);
	       else tsr(i,j,k,l) = tsr(j,i,k,l) = tsr(i,j,l,k) = factor * mat(p,q++);

  return tsr;
}


//----------------------------------------------------------------
template <class T> matrix<T> tensor2matrix (const tensor4<T>& tsr)
//----------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2() && tsr.dim3() && tsr.dim4());

  matrix<T> mat(tsr.dim1()*tsr.dim2(),tsr.dim3()*tsr.dim4(),false);

  for (int i=1,p=1; i<=tsr.dim1(); i++)
    for (int j=1; j<=tsr.dim2(); j++,p++)
      for (int k=1,q=1; k<=tsr.dim3(); k++)
	for (int l=1; l<=tsr.dim4(); l++)
	  mat (p,q++) = tsr(i,j,k,l);

  return mat;
}


//-------------------------------------------------------------------
template <class T> vector<T> symtensor2vector (const tensor2<T>& tsr)
//-------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2());
  assert (tsr.dim1() == tsr.dim2());

  long double factor = sqrt( (long double) 2);
  int n = tsr.dim1();
  vector<T> v(.5*n*(n+1),false);

  for (int i=1,p=1; i<=n; i++)
    { v[p++] = tsr(i,i);
      for (int j=i+1; j<=n; j++)
	{ assert (abs(tsr(i,j) - tsr(j,i)) < epsilon);
	  v[p++] = factor * tsr(i,j);
    }   }

  return v;
}


//-------------------------------------------------------------------
template <class T> matrix<T> symtensor2matrix (const tensor4<T>& tsr)
//-------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2() && tsr.dim3() && tsr.dim4());
  assert (tsr.dim1() == tsr.dim2());
  assert (tsr.dim3() == tsr.dim4());

  long double factor = sqrt( (long double) 2);
  int nb1 = tsr.dim1();
  int nb2 = tsr.dim3();
  matrix<T> mat(.5*nb1*(nb1+1),.5*nb2*(nb2+1),false);

  for (int i=1,p=1; i<=nb1; i++)
    for (int j=i; j<=nb1; j++,p++)
      for (int k=1,q=1; k<=nb2; k++)
	for (int l=k; l<=nb2; l++)
       	  { assert (abs(tsr(i,j,k,l) - tsr(i,j,l,k)) < epsilon);
	    assert (abs(tsr(i,j,k,l) - tsr(j,i,k,l)) < epsilon);
	    assert (abs(tsr(i,j,k,l) - tsr(j,i,l,k)) < epsilon);
	    if (i==j && k==l) mat (p,q++) = tsr(i,j,k,l);
	    else if (i!=j && k!=l) mat(p,q++) = 2 * tsr(i,j,k,l);
	         else mat(p,q++) = factor * tsr(i,j,k,l);
	  }

  return mat;
}


#endif
